﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.ComponentModel.Composition.Hosting;
using System.Diagnostics;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace OGSNET
{
    using OGSNET.Plugin;

    /** \brief プラグインを扱うクラス */
    public class PluginControl
    {
        private DirectoryCatalog catalog;       /**< カタログ */
        private CompositionContainer container; /**< コンテナ */

        /** \arg \c directory プラグインを探すディレクトリ */
        public PluginControl(string directory)
        {
            this.catalog   = new DirectoryCatalog(directory);
            this.container = new CompositionContainer(this.catalog);
        }

        /**
         * \brief 指定した配列から PluginId を指定してプラグインを取得する
         * 
         * 該当するプラグインが複数存在する場合は、バージョンを比較し一番新しいものを取得する。
         * 該当するプラグインが存在しない場合は、null を返す。
         * 
         * \arg \c plugins プラグインの配列
         * \arg \c pluginId プラグインを識別するID
         */
        public static Lazy<T, IPluginMetaData> GetPlugin<T>(Lazy<T, IPluginMetaData> [] plugins, string pluginId)
            where T : IPlugin
        {
            var temp = new List<Lazy<T, IPluginMetaData>>();

            foreach (var plugin in plugins)
            {
                if (plugin.Metadata.PluginId == pluginId)
                {
                    temp.Add(plugin);
                }
            }

            if (temp.Count == 0)
            {
                return null;
            }

            temp.Sort();

            return temp.Last();
        }

        /** \brief 指定した配列から PluginId を指定してプラグインを取得する */
        public static Lazy<IPlugin, IPluginMetaData> GetPlugin(Lazy<IPlugin, IPluginMetaData>[] plugins, string pluginId)
        {
            return PluginControl.GetPlugin<IPlugin>(plugins, pluginId);
        }

        /** \brief 型を指定してすべてのプラグインを取得 */
        public Lazy<T, IPluginMetaData>[] GetPlugins<T>() where T : IPlugin
        {
            var exports = this.container.GetExports<T, IPluginMetaData>();

            return exports.ToArray();
        }

        /** 
         * \brief すべてのプラグインを取得
         * \ref GetPlugins
         */
        public Lazy<IPlugin, IPluginMetaData>[] GetPlugins()
        {
            return this.GetPlugins<IPlugin>();
        }

        /** \brief 型と PluginId を指定してプラグインを取得する */
        public Lazy<T, IPluginMetaData> GetPlugin<T>(string pluginId) where T : IPlugin
        {
            return PluginControl.GetPlugin<T>(this.GetPlugins<T>(), pluginId);
        }

        /** \brief PluginId を指定してプラグインを取得する */
        public Lazy<IPlugin, IPluginMetaData> GetPlugin(string pluginId)
        {
            return this.GetPlugin<IPlugin>(pluginId);
        }
    }

    /**
     * \brief プラグインを開始する
     * 
     * プラグインの実行が終了した、もしくはタイムアウトになったらリソースを解放する
     */
    public class PluginStarter : IDisposable
    {
        protected PluginStartInfo PluginStartInfo;
        private bool IsDispose = false;

        /**
         * \brief プラグインを開始する
         * \arg \c plugin 開始するプラグイン
         * \arg \c psi プラグインへ渡す情報
         * \arg \c timeout プラグインのタイムアウト (msec)
         */
        public PluginStarter(IPlugin plugin, PluginStartInfo psi)
        {
            this.PluginStartInfo = psi;
            plugin.Start(psi);
        }

        /** \brief 解放処理 */
        public void Dispose()
        {
            if (!this.IsDispose)
            {
                try
                {
                    this.PluginStartInfo.Browser.Dispose();
                    this.IsDispose = true;
                }
                catch (Exception) { }
            }
        }

        /** \brief 解放処理がされていなかったら解放 */
        ~PluginStarter()
        {
            if (!this.IsDispose)
            {
                this.Dispose();
            }
        }
    }
}
